
#ifndef __LIGHT_ITERATOR_STRUCT
#define __LIGHT_ITERATOR_STRUCT
struct Light_iterator {
    float3 _point;
    float4 contribution;
    float4 raw_contribution;
    float  dot_nl;
    float3 direction;	// from surface to light
    float  distance;
    float4 shadow;
    int    count;
};
#endif // __LIGHT_ITERATOR_STRUCT


// For standard lights, position and direction are defined in view space
struct MAXLight_Omni {
	float4 color;
	float4 position;
	float4 falloffDistance;	// distance_scale/distance_falloff_start/distance_falloff_limit/distance_falloff_exponent
};

struct MAXLight_Spot {
	float4 color;
	float4 position;
	float4 direction;
	float4 falloffDistance;	// distance_scale/distance_falloff_start/distance_falloff_limit/distance_falloff_exponent
	float4 falloffSpread;	// light_spread_inner/light_spread_falloff/max_falloff/max_falloff_cos
};

struct MAXLight_Dir {
	float4 color;
	float4 position;
	float4 direction;	
};

struct MAXLight_Photometric {
	float4 posWorldToLight[3];
	float4 norWorldToLight[3];
	float4 color;
	float4 posWorld;
};


void TransformToLightSpace(
	float3 surfacePosWorld,
	float3 surfaceNorWorld,
	float4 transPos[3],			// world to light space transformation for position
	float4 transNor[3],			// world to light space transformation for normal
	out float3 surfacePosLight, 
	out float3 surfaceNorLight)
{
	surfacePosLight.x = dot(transPos[0], float4(surfacePosWorld, 1.0));
	surfacePosLight.y = dot(transPos[1], float4(surfacePosWorld, 1.0));
	surfacePosLight.z = dot(transPos[2], float4(surfacePosWorld, 1.0));

	surfaceNorLight.x = dot(transNor[0], float4(surfaceNorWorld, 0.0));
	surfaceNorLight.y = dot(transNor[1], float4(surfaceNorWorld, 0.0));
	surfaceNorLight.z = dot(transNor[2], float4(surfaceNorWorld, 0.0));
}
	

// Omni-direction light, shading in world space
void stdlight_omni(
	float3 surfacePosWorld, 
	float3 surfaceNorWorld, 
	out Light_iterator __light, 
	MAXLight_Omni omniLight)
{
	__light._point = omniLight.position;
	__light.direction = omniLight.position - surfacePosWorld;
	__light.distance = length(__light.direction);
	__light.direction /= __light.distance;

	// Attenuation due to distance	
	float4 falloffDistance;	
	
	float distance_scale = omniLight.falloffDistance.x;
	float distance_falloff_start = omniLight.falloffDistance.y;
	float distance_falloff_limit = omniLight.falloffDistance.z;
	float distance_falloff_exponent = omniLight.falloffDistance.w;
	
	float d = saturate(
		(__light.distance/(distance_scale - distance_falloff_start))/
		(distance_falloff_limit - distance_falloff_start));
	float fAttenuation = 1.0/(1.0 + pow(d, distance_falloff_exponent));

	float fDistribution = 1.0;

	__light.dot_nl = saturate(dot(normalize(__light.direction), surfaceNorWorld));
	__light.contribution = omniLight.color * fDistribution * fAttenuation;
	__light.raw_contribution = __light.contribution;
	
	__light.shadow = 1.0;
	__light.count = 0;
}

// Directional light, shading in world space
void stdlight_dir(
	float3 surfacePosWorld, 
	float3 surfaceNorWorld, 
	out Light_iterator __light, 
	MAXLight_Dir dirLight)
{
	__light._point = dirLight.position;
	__light.direction = -dirLight.direction;
	__light.distance = saturate(dot(surfacePosWorld - dirLight.position, dirLight.direction));

	float fAttenuation = 1.0;
	float fDistribution = 1.0;

	__light.dot_nl = dot(-dirLight.direction, surfaceNorWorld);
	__light.contribution = dirLight.color * fDistribution * fAttenuation;
	__light.raw_contribution = __light.contribution;
	
	__light.shadow = 1.0;
	__light.count = 0;
}


// Spot light, shading in world space
void stdlight_spot(
	float3 surfacePosWorld, 
	float3 surfaceNorWorld, 
	out Light_iterator __light, 
	MAXLight_Spot spotLight)
{
	__light._point = spotLight.position;
	__light.direction = spotLight.position - surfacePosWorld;
	__light.distance = length(__light.direction);
	__light.direction /= __light.distance;

	// Attenuation 
	float distance_scale = spotLight.falloffDistance.x;
	float distance_falloff_start = spotLight.falloffDistance.y;
	float distance_falloff_limit = spotLight.falloffDistance.z;
	float distance_falloff_exponent = spotLight.falloffDistance.w;
	
	float d = saturate(
		(__light.distance/(distance_scale - distance_falloff_start))/
		(distance_falloff_limit - distance_falloff_start));
	float fAttenuation = 1.0/(1.0 + pow(d, distance_falloff_exponent));
	
	// Distribution due to spread angle
	float light_spread_inner = spotLight.falloffSpread.x;
	float light_spread_falloff = spotLight.falloffSpread.y;
	float max_falloff = spotLight.falloffSpread.z;
	float max_falloff_cos =spotLight.falloffSpread.w;
	
	float fDistribution = 1.0;
	float3 lightDirection = __light.direction;
	
	// spread falloff
	float locationSpread = dot(-lightDirection, spotLight.direction);
	if( locationSpread < max_falloff_cos ) {
		fDistribution = 0;
	}
	else {
		float locationSpreadRadians = acos(locationSpread);
		float light_spread_innerRadians = radians(light_spread_inner);
		if( locationSpreadRadians < light_spread_innerRadians ) {
			fDistribution = 1.f;
		}
		else {
			fDistribution = pow(1 - 
				saturate(locationSpreadRadians - light_spread_innerRadians)/
				(max_falloff - light_spread_innerRadians),
				light_spread_falloff);
		}
	}
	
	__light.dot_nl = saturate(dot(lightDirection, surfaceNorWorld));
	__light.contribution = spotLight.color * fAttenuation * fDistribution;
	__light.raw_contribution = __light.contribution;
	
	__light.shadow = 1.0;
	__light.count = 0;
}



// Photometric light (point shape, omni distribution), shading in light space
void pmlight_point_omni(
	float3 surfacePosWorld, 
	float3 surfaceNorWorld, 
	out Light_iterator __light,
	MAXLight_Photometric pmLight)
{
	// Both in light space
	float3 surfacePos;	
	float3 surfaceNor;
	
	TransformToLightSpace(
		surfacePosWorld, 
		surfaceNorWorld, 
		pmLight.posWorldToLight, 
		pmLight.norWorldToLight,
		surfacePos,
		surfaceNor);

	// World space
	__light._point = pmLight.posWorld;
	__light.direction = pmLight.posWorld - surfacePosWorld;
	__light.distance = length(__light.direction);
	__light.direction /= __light.distance;

	float fAttenuation = 1.0 / (__light.distance * __light.distance);
	float fDistribution = 1.0;

	// Light space
	__light.dot_nl = saturate(dot(normalize(-surfacePos), surfaceNor));
	__light.contribution = pmLight.color * fDistribution * fAttenuation;
	__light.raw_contribution = __light.contribution;
	
	__light.shadow = 1.0;
	__light.count = 0;
}




/*
// Photometric light (disc shape, omni distribution)
void pmlight_disc_omni(
	float3 surfacePosWorld, 
	float3 surfaceNorWorld, 
	out Light_iterator __light,
	MAXLight_Photometric pmLight)
{
	float3 posLight;
	float3 norLight;
	
	posLight.x = dot(pmLight.posWorldToLight[0], float4(surfacePosWorld, 1.0));
	posLight.y = dot(pmLight.posWorldToLight[1], float4(surfacePosWorld, 1.0));
	posLight.z = dot(pmLight.posWorldToLight[2], float4(surfacePosWorld, 1.0));

	norLight.x = dot(pmLight.norWorldToLight[0], float4(surfaceNorWorld, 0.0));
	norLight.y = dot(pmLight.norWorldToLight[1], float4(surfaceNorWorld, 0.0));
	norLight.z = dot(pmLight.norWorldToLight[2], float4(surfaceNorWorld, 0.0));

	// World space
	__light._point = pmLight.surfacePosWorld;
	__light.direction = surfacePosWorld - pmLight.surfacePosWorld;
	__light.distance = length(__light.direction);
	__light.direction /= __light.distance;

	float fDistribution = 1.0;


	// Light space
	
	float fRadius = pmLight.shape.x;
	float fInversedRadiusThresh = pmLight.shape.y;
	float fDistanceAttenPower = pmLight.shape.z;
	float fProjectionAttenPower = pmLight.shape.w;

	// Find on the light source the nearest point to the target point
	float3 lightPosition = float3(posLight.xy, 0);
	// Clamp the point to be inside the disc
	float fTempRadius = length(lightPosition);
	lightPosition = lightPosition / fTempRadius * min(fTempRadius, fRadius);

	// The attenuation is calculated in an empirical way	
	float fAttenuation = pow(__light.distance, fDistanceAttenPower) * 
		pow(min(1.0, abs(posLight.z) * fInversedRadiusThresh), fProjectionAttenPower);

	__light.dot_nl = saturate(dot(normalize(lightPosition-posLight), norLight));
	__light.contribution = pmLight.color * fDistribution * fAttenuation;
	__light.raw_contribution = __light.contribution;
	
	__light.shadow = 1.0;
	__light.count = 0;
}
*/

